package MyTemplate;

use strict;
use utf8;
use Data::Dumper;
use JSON;

###############################################################################
sub new {
	my ($class,$sql,$locale)=@_;
	my $self={
		sql=>$sql,
		paths=>["templates"],
		locale=>$locale||"default"
	};
	bless ($self,$class);
	return $self;
}

###############################################################################
sub save_error {
	my ($tmpl,$space,$query,$error)=@_;
	return $error unless open OUT,">>../logs/mytemplate.log";
	print OUT localtime()."\n";
	print OUT "uid=".($space->{uid}||0)." action=".($space->{action}||"")."\n";
	print OUT $error;
	print OUT "\n";
	close OUT;
	return $error;
}

###############################################################################
sub load {
	my ($self,$file)=@_;
	my $result;
	foreach (@{$self->{paths}}) {
		next if defined $result;
		if (open IN,"$_/$file.html") {
			{local $/;$result=<IN>;}
			close IN;
			Encode::_utf8_on($result);
		}
	}
	unless (defined $result) {
		$result="<b>[$file]</b>";
	}
	$self->{templates}->{$file}=$self->parse($result,$file);
}

###############################################################################
sub parse_path {
	my ($self,$path)=@_;
	return $path unless $path=~/\//;
	my $aref=[];
	while ($path=~s/^(.+?)\///) {push(@$aref,$1);}
	push(@$aref,$path);
	return $aref;
}

###############################################################################
sub parse {
	my ($self,$text,$file)=@_;
	my $r=[];
	my $oldr=[];
	while ($text=~s/^(.*?)#([,\-\|\:\w\/ \(\)\{\}"]*)#//s) {
		my ($prefix,$tag)=($1,$2);
		push(@$r,["text",$prefix]) if length($prefix);
		if ($tag eq "") {
			push(@$r,["text","#"]);
		} elsif ($tag=~/^Dumper$/) {
			push(@$r,["Dumper"]);

		} elsif ($tag=~/^each clean$/) {
			push(@$r,["each-clean"]);

		} elsif ($tag=~/^\/(if|unless|foreach|each|set|ifeq|unlesseq|switch)$/) {
			my $tag=$1;
			$r=pop(@$oldr);
			die $self->save_error({},undef,"Closing $tag without opening one") unless $r;
			my $rs=@$r;
			my $opener=$r->[$rs-1]->[0];
			$opener=~s/Q//;
			if ($opener ne $tag && $opener ne $tag."eq") {
				$r=[["text","<b>Opening and closing tags mismatch: #$opener# and #/$tag#</b>"]];
				return $r;
			}
		
		} elsif ($tag=~/^((?:\w+(?:\([^()]*\))?:)*)([\w\/]+)$/) {
			my ($params,$path)=(defined $1?$1:"",$2);
			my $data=["var",$self->parse_path($path)];
			if ($params=~s/^Q://) {
				$data->[0]="varQ";
			} elsif ($params=~s/^QI://) {
				$data->[0]="varQ";
				push(@$data,"I",undef);
			}
			while ($params=~s/^(\w+)((?:\([^()]*\))?)://) {
				my ($a,$b)=($1,$2);
				$b=~s/^\(//;
				$b=~s/\)$//;
				push(@$data,$a,$b);
			}
			push(@$r,$data);
			
		} elsif ($tag=~/^(\w+)\(([,\w\/]*)\)$/) {
			push(@$r,["func",$1,$2]);
		
		} elsif ($tag=~/^(\w+)\(([\w\/]*)\)\{([\w\/]+)\}$/) {
			push(@$r,["funclist",$1,$2,$self->parse_path($3)]);
		
		} elsif ($tag=~/^([\w\/\-]+)\{([\w\/]*)\}$/) {
			push(@$r,["tmpl",$1,$2?$self->parse_path($2):undef]);
		
		} elsif ($tag=~/^foreach \{([\w\/]+)\}$/) {
			my $ri=[];
			push(@$r,["foreach",$self->parse_path($1),undef,$ri]);
			push(@$oldr,$r);
			$r=$ri;
		
		} elsif ($tag=~/^foreach (\w+) \{([\w\/]+)\}$/) {
			my $ri=[];
			push(@$r,["foreach",$self->parse_path($2),$1,$ri]);
			push(@$oldr,$r);
			$r=$ri;

		} elsif ($tag=~/^each (\d+):(\d*)$/) {
			my $ri=[];
			push(@$r,["each",$1,$2,$ri]);
			push(@$oldr,$r);
			$r=$ri;
		
		} elsif ($tag=~/^each ([\w\/]+):(\d*)$/) {
			my $ri=[];
			push(@$r,["each",$self->parse_path($1),$2,$ri]);
			push(@$oldr,$r);
			$r=$ri;

		} elsif ($tag=~/^(if|unless) ([\w\/]+)$/) {	
			my $ri=[];
			push(@$r,[$1,$self->parse_path($2),$ri]);
			push(@$oldr,$r);
			$r=$ri;
		
		} elsif ($tag=~/^switch ([\w\/]+)$/) {	
			my $ri=[];
			push(@$r,["switch",$self->parse_path($1),{}]);
			push(@$oldr,$r);
			$r=$ri;
		
		} elsif ($tag=~/^case "([^"]*)"$/) {
			my $case=$1;
			$r=[];
			my $or=$oldr->[@$oldr-1];
			my $st=$or->[@$or-1]->[0];
			
			unless ($st eq "switch") {
				$r=[["text","<b>(Evaluating #case#) - not a #switch# statement, is a #$st# statement</b><br><pre>".Dumper($oldr)."</pre>"]];
				return $r;
			}
			$or->[@$or-1]->[2]->{$case}=$r;
		
		} elsif ($tag=~/^(if|unless) ([\w\/]+) eq "([^"]*)"$/) {	
			my $ri=[];
			push(@$r,[$1."eq",$self->parse_path($2),$3,$ri]);
			push(@$oldr,$r);
			$r=$ri;

		} elsif ($tag=~/^set ([\w\/]+)$/) {	
			my $ri=[];
			push(@$r,["set",$self->parse_path($1),$ri]);
			push(@$oldr,$r);
			$r=$ri;
	
		} elsif ($tag=~/^(if|unless) Q:([\w\/]+)$/) {
			my $ri=[];
			push(@$r,[$1."Q",$self->parse_path($2),$ri]);
			push(@$oldr,$r);
			$r=$ri;
		} else {
			die $self->save_error({},undef,"Can not parse #$tag#, file $file\n");
		}
	}
	push(@$r,["text",$text]) if length($text);
	if (@$oldr) {
		my $a="";
		while (@$oldr) {
			my $t=pop @$oldr;
			$a=", " if $a;
			my $ts=@$t;
			my $tag=$t->[$ts-1]->[0];
			$a.="#$tag#";
		}
		$r=[["text","<b>Tags left unclosed: $a</b>"]];
		return $r;
	}
#	die Dumper($r);
	return $r;
}

###############################################################################
sub process {
	my ($self,$file,$space,$query)=@_;
	$self->load($file) unless $self->{templates}->{$file};
	return $self->eval_arr($self->{templates}->{$file},$space,$query);
}

###############################################################################
sub eval_path {
	my ($self,$space,$path)=@_;
	if (ref $path) {
		foreach (@$path) {
			return "[!NOREF!]" unless ref $space;
			$space=$space->{$_};
			return undef unless defined $space;
		}
		return $space;
	} else {
		return $space->{$path};
	}
}

###############################################################################
sub eval_path_save {
	my ($self,$space,$path,$data)=@_;
	if (ref $path) {
		my $i=0;
		while ($path->[$i+1]) {
			$space=$space->{$path->[$i]};
			$i++;
		}
		$space->{$path->[$i]}=$data;
	} else {
		$space->{$path}=$data;
	}
}

###############################################################################
sub eval_func_clean_html {
	my ($tmpl,$space,$query,$var)=@_;
	$var=~s/<.*?>//sg;
	return $var;
}

###############################################################################
sub eval_func_I {
	my ($self,$space,$query,$var,$i1)=@_;
	return $self->text_input($var);
}

###############################################################################
sub eval_func_JS {
	my ($self,$space,$query,$var,$i1)=@_;
	$var=~s/\\/\\\\/g;
	$var=~s/"/\\"/g;
	$var=~s/\n/\\n/g;
	$var=~s/\r/\\r/g;
	return $var;
}

###############################################################################
sub eval_func_B {
	my ($self,$space,$query,$var,$i1)=@_;
	return $self->text_br($var);
}

###############################################################################
sub eval_func_MUL {
	my ($self,$space,$query,$var,$i1)=@_;
	return $var*$i1;
}

###############################################################################
sub eval_func_MUL1 {
	my ($self,$space,$query,$var,$i1)=@_;
	return ($var-1)*$i1;
}

###############################################################################
sub eval_func_T {
	my ($self,$space,$query,$var,$i1)=@_;
	return $self->text($var);
}

###############################################################################
sub eval_func_preview {
	my ($self,$space,$query,$var,$i1)=@_;
	$var=~s/\.(jpg|gif|png)/-preview$i1.$1/;
	return $var;
}

###############################################################################
sub eval_func_O {
	my ($self,$space,$query,$var,$i1)=@_;
	return $self->mk_options($space,$var);
}

###############################################################################
sub eval_func_CHK {
	my ($self,$space,$query,$var,$i1)=@_;
	return $var?" checked=\"yes\"":"";
}

###############################################################################
sub eval_func_C {
	my ($self,$space,$query,$var,$i1)=@_;
	return $self->crop($var,$i1);
}

###############################################################################
sub eval_func_noslashes {
	my ($self,$space,$query,$var,$i1)=@_;
	$var=~s/\///g;
	return $var;
}

###############################################################################
sub eval_func_http {
	my ($self,$space,$query,$var,$i1)=@_;
	$var="http://$var" unless $var=~/^http:\/\//;
	return $var;
}

###############################################################################
sub eval_func_dumper {
	my ($self,$space,$query,$var,$i1)=@_;
	return $self->dumper($var);
}

###############################################################################
sub eval_func_process {
	my ($self,$space,$query,$var,$i1)=@_;
	my $space2=$i1?$space->{$i1}:$space;
	return "" unless defined $var && $var ne "";
	return $self->tmpl_apply($var,$space2,$query);
}

###############################################################################
sub eval_func_path {
	my ($self,$space,$query,$var,$i1)=@_;
	$var=~/^(.*\/)/;
	return defined $1?$1:"";
}

###############################################################################
sub eval_func_format {
	my ($self,$space,$query,$var,$fmt)=@_;
	my $sign="";
	if ($var<0) {$sign="-";$var=-$var;}
	my $var1=sprintf('%f',$var);
	my $float=0;
	$float=$1 if $var1=~s/(\.\d+)$//;
	my $int=$var1;
#	my $int=int($var);
#	my $float=$var-$int;
	if ($fmt=~s/G//) {
		my $src=$int;
		$int="";
		while ($src=~s/(\d\d\d)$//) {
			$int=" ".$int if $int ne "";
			$int=$1.$int;
		}
		$int=" ".$int if $int ne "";
		$int=$src.$int;
	}
	return $sign.$int if $fmt==0;
	$float=sprintf('%.'.$fmt.'f',$float);
	$float=~s/^.*\././;
	return $sign.$int.$float;
}


###############################################################################
sub eval_func_extention {
	my ($self,$space,$query,$var,$i1)=@_;
	if ($var=~/\.(\w{1,6})$/) {
		return $1;
	}
	return "";
}

###############################################################################
sub eval_func_rus {
	my ($self,$space,$query,$var,$i1)=@_;
	my @a=split /,/,$i1;
	my $n=$var%100;
	my $nn=$n%10;
	if ($n>10 && $n<19) {
		return $a[2];
	} elsif ($nn==1) {
		return $a[0];
	} elsif ($nn==2) {
		return $a[1];
	} elsif ($nn==3) {
		return $a[1];
	} elsif ($nn==4) {
		return $a[1];
	}
	return $a[2];
}

###############################################################################
sub eval_func_mkpreviews {
	my ($self,$space,$query,$var,$i1)=@_;
	$var=~s/(<img [^>]*preview[^>]*>)/$self->mk_img_to_preview($1,$i1)/ge;
	return $var;
}

###############################################################################
sub eval_func_CROP {
	my ($tmpl,$space,$query,$text,$size)=@_;
	$text=~s/<\/?(\w+).*?>/($1 eq "br" || $1 eq "BR")?"<br>":""/ge;
	return $text if length($text)<$size;
	my $crop=substr($text,0,$size);
	if ($crop=~/<[^>]*$/) {
		$crop.=">";
	}
	$crop.="...";
	return "<div class='crop' title=\"".$tmpl->text_input($text)."\" alt=\"".$tmpl->text_input($text)."\">".$crop."</div>";
}

###############################################################################
sub JSON {
	my ($tmpl,$var)=@_;
	return to_json($var);
}

###############################################################################
sub eval_func_JSON {
	my ($tmpl,$space,$query,$var)=@_;
	return to_json($var);
}

###############################################################################
sub eval_arr {
	my ($self,$arr,$space,$query)=@_;
	my $output="";
	my $var;
	foreach my $i (@$arr) {
		if ($i->[0] eq "text") {
			$output.=$i->[1];
		
		} elsif ($i->[0] eq "var" || $i->[0] eq "varQ") {
			if ($i->[0] eq "var") {
				$var=$self->eval_path($space,$i->[1]);
			} else {
				$var=$query->param($i->[1]);
				Encode::_utf8_on($var);
			}
			if (defined $var) {
				my $j=2;
				while (@$i>$j) {
					my $evalname="eval_func_".$i->[$j];
					if ($self->can($evalname)) {
						$var=$self->$evalname($space,$query,$var,$i->[$j+1]);
					} else {
						$var="<b>[unknown var tag ".$i->[$j]."]</b>";
					}
					$j+=2;
				}
				$output.=$var if defined $var;
			}
		
		} elsif ($i->[0] eq "Dumper") {
			$var="<pre>".$self->text(Dumper($space))."</pre>";
#			$var=~s/\n/<br>\n/g;
			$output.=$var;
		
		} elsif ($i->[0] eq "func") {
			my $a=$self->subprocess($space,$i->[1],$query,$i->[2]);
			$output.=$a if defined $a;
		
		} elsif ($i->[0] eq "funclist") {
			$output.=$self->subprocess_list($space,$i->[1],$i->[2],$query,$i->[3]);
		
		} elsif ($i->[0] eq "foreach") {
			my $tmp=$query->param("foreach_rownum");
			$var=$self->eval_path($space,$i->[1]);
			if (ref($var) eq "ARRAY") {
				$self->replace($query,"foreach_rownum",0);
				foreach my $tt(@$var) {
					if (defined $i->[2]) {
						$space->{$i->[2]}=$tt;
						$output.=$self->eval_arr($i->[3],$space,$query);
					} else {
						$output.=$self->eval_arr($i->[3],$tt,$query);
					}
					$self->replace($query,"foreach_rownum",$query->param("foreach_rownum")+1);
				}
			} else {
				$self->replace($query,"foreach_rownum",0);
				if (defined $i->[2]) {
					$space->{$i->[2]}=$var;
					$output.=$self->eval_arr($i->[3],$space,$query);
				} else {
					$output.=$self->eval_arr($i->[3],$var,$query);
				}
			}
			delete $space->{$i->[2]} if defined $i->[2];
			$self->replace($query,"foreach_rownum",$tmp);

		} elsif ($i->[0] eq "tmpl") {
			$var=$i->[2]?$self->eval_path($space,$i->[2]):$space;
			if (ref($var) eq "ARRAY") {
				foreach my $tt(@$var) {
					$output.=$self->process($i->[1],$tt,$query);
				}
			} else {
				$output.=$self->process($i->[1],$var,$query);
			}
		
		} elsif ($i->[0] eq "set") {
			my $result=$self->eval_arr($i->[2],$space,$query);
			$self->eval_path_save($space,$i->[1],$result);
		
		} elsif ($i->[0] eq "switch") {
			$var=$self->eval_path($space,$i->[1]);
			$var="" unless defined $var;
			if ($i->[2]->{$var}) {
				$output.=$self->eval_arr($i->[2]->{$var},$space,$query);
			} elsif ($i->[2]->{""}) {
				$output.=$self->eval_arr($i->[2]->{""},$space,$query);
			}
#			$output.="<pre>".Dumper($i)."</pre>";

		} elsif ($i->[0] eq "if") {
			$var=$self->eval_path($space,$i->[1]);
			if ($var && (ref $var ne "ARRAY" || @$var)) {
				$space->{_}=$var;
				$output.=$self->eval_arr($i->[2],$space,$query);
			}

		} elsif ($i->[0] eq "unless") {
			$var=$self->eval_path($space,$i->[1]);
			$output.=$self->eval_arr($i->[2],$space,$query) if (!$var) || (ref $var eq "ARRAY" && @$var==0);
		
		} elsif ($i->[0] eq "ifeq" || $i->[0] eq "unlesseq") {
			$var=$self->eval_path($space,$i->[1]);
			if (!ref $var) {
				$var="" unless defined $var;
				$output.=$self->eval_arr($i->[3],$space,$query) if $i->[0] eq "ifeq" && $var eq $i->[2];
				$output.=$self->eval_arr($i->[3],$space,$query) if $i->[0] eq "unlesseq" && $var ne $i->[2];
			}

		} elsif ($i->[0] eq "each-clean") {
			$self->replace($query,"foreach_rownum",-1);
		} elsif ($i->[0] eq "each") {
			my $flag=0;
			my $i1=$i->[1];
			my $i2=$i->[2];
			if (ref $i1) {
				$i1=$self->eval_path($space,$i1);
			}
			if ($i2 eq "") {
				$i2=$i1-1;
			}
			if ($i1<1) {
				$output.="<b>[each error: '$i1':'$i2']</b>";
			} else {
				if (defined $query->param("foreach_rownum")) {
					$flag=$query->param("foreach_rownum")%$i1==$i2?1:0;
				}
				$output.=$self->eval_arr($i->[3],$space,$query) if $flag;
			}
		
		} elsif ($i->[0] eq "ifQ") {
			$output.=$self->eval_arr($i->[2],$space,$query) if $query->param($i->[1]);
		
		} elsif ($i->[0] eq "unlessQ") {
			$output.=$self->eval_arr($i->[2],$space,$query) unless $query->param($i->[1]);
		
		} else {
			die $self->save_error($space,$query,"process()");
		}
	}
	return $output;
}

###############################################################################
sub mk_img_to_preview {
	my ($self,$imgtext,$params)=@_;
#	return $imgtext if $imgtext=~/-preview/;
	return $imgtext unless $imgtext=~/^(.*src=)['"](.*?)['"](.*)$/;
	my ($prefix,$file,$suffix)=($1,$2,$3);
	$file=~s/-preview(\d+|\d+\w+\d*)\./\./;
	my $sml1=$1;
	return $imgtext unless $file=~/^(.*)\.(gif|png|jpg)$/;
	my($name,$ext)=($1,$2);
	return $imgtext unless $params=~/^(\d*(?:\w+\d*)?),(\d+(?:\w+\d*)?)((?:,title)?)/;
	my($sml,$big,$hastitle)=($1||$sml1,$2,$3);
	my $style1="";
	my $styleadd="cursor:pointer;border: 1px solid #808080; margin-right: 10px;";
	if ($prefix=~s/style="(.*?)"/style="$styleadd$1"/) {
	} elsif ($suffix=~s/style="(.*?)"/style="$styleadd$1"/) {
	} else {
		$style1="style=\"$styleadd\"";
	}
	my $r="$prefix\"$name-preview$sml.$ext\" onclick='gallery_open(this,\"preview$big\")' $style1".$suffix;
	$r=~s/width=".*?"//;
	$r=~s/height=".*?"//;
	if ($hastitle) {
		return $r unless $r=~/align=['"](.*?)['"]|style="(.*?float:.*?)"/;
		my ($align,$style)=($1,$2);
		return $r unless $r=~/alt=['"](.*?)["']/;
		$style="float: $align" unless $style;
		my $text=$1;
		$style=~s/border: .*?;//;
		$r=~s/align="\w+"//;
		$r="<div style=\"margin: 10px;$style\">$r<br>$text</div>";
	}
	return $r;
}

###############################################################################
sub mk_options {
	my ($self,$space,$var)=@_;
#	my $var=$self->eval_path($space,$name);
	my $r="";
	foreach my $i (@$var) {
		my ($id,$level,$name);
		if (defined $i->{name}) {
			($id,$level,$name)=($i->{id},$i->{level},$i->{name});
		} else {
			($id,$level,$name)=($i->{ID},$i->{LEVEL},$i->{NAME});
		}
		my $spacer="";
		$spacer="&nbsp;&nbsp;&nbsp;"x($level-1) if $level;
		$r.="<option value=\"$id\">$spacer$name</option>";
	}
	return $r;
}

###############################################################################
sub text {
	my ($self,$a)=@_;
	return "" unless defined $a;
	$a=~s/&/&amp;/g;
	$a=~s/</&lt;/g;
	$a=~s/>/&gt;/g;
	return $a;
}

###############################################################################
sub text_br {
	my ($self,$a)=@_;
	return "" unless defined $a;
	$a=~s/&/&amp;/g;
	$a=~s/</&lt;/g;
	$a=~s/>/&gt;/g;
	$a=~s/\n/<br>\n/g;
	return $a;
}

###############################################################################
sub text_input {
	my ($self,$a)=@_;
	return "" unless defined $a;
	$a=~s/&/&amp;/g;
	$a=~s/</&lt;/g;
	$a=~s/>/&gt;/g;
	$a=~s/"/&quot;/g;
	$a=~s/\n/\\n/g;
	return $a;
}

###############################################################################
sub subprocess_list {
	my ($self,$space,$func,$param,$query,$path)=@_;
	my $var=$self->eval_path($space,$path);
	return "" unless defined $var;
	my $ref=ref $var;
	return "!NOT A LIST!" unless $ref eq "ARRAY";
	my $r="";
	foreach my $i (@$var) {
		$r.=$self->subprocess($i,$func,$query,$param);
	}
	return $r;
}

###############################################################################
sub subprocess {
	my ($self,$space,$func,$query,$param)=@_;
	my $ret;
	eval '$ret=$self->func_'.$func.'($space,$query,$param);';
	if ($@) {
		my $space1={error=>'<pre>'.$self->text($@).'</pre>'};
		$self->save_error($space,$query,$@);
		return $self->process("error",$space1,$query);
	}
	return $ret;
}

###############################################################################
sub replace {
	my ($self,$query,$param,$value)=@_;
	$query->delete($param);
	$query->append(-name=>$param,-values=>[$value]);
}

###############################################################################
sub crop {
	my ($self,$text,$size)=@_;
	$text=~s/<\/?(\w+).*?>/($1 eq "br" || $1 eq "BR")?"<br>":""/ge;
	return $text if length($text)<$size;
	my $crop=substr($text,0,$size);
	if ($crop=~/<[^>]*$/) {
		$crop.=">";
	}
	$crop.="...";
	return $crop;
}

###############################################################################
sub crop_text {
	my ($self,$text,$size)=@_;
	return $self->text($text) if length($text)<$size;
	my $crop=substr($text,0,$size);
	$crop.="...";
	return $self->text($crop);
}

###############################################################################
sub tmpl_apply {
	my ($tmpl,$text,$space,$query,$alias)=@_;
	return "" unless defined $text;
	return "" if $text eq "";
	$alias||="(pseudo) ".$text;
	$tmpl->{templates}->{$alias}=$tmpl->parse($text,$alias) unless $tmpl->{templates}->{$alias};
#	die Dumper($tmpl->{templates}->{$text}) if $text=~/#/;
	return $tmpl->process($alias,$space,$query);
}

###############################################################################
sub locale {
	my($self,$fullalias)=@_;
	my $locale=$self->{locale};
	return $self->{locales}->{$locale}->{$fullalias} if $self->{locales}->{$locale}->{$fullalias};
	$fullalias=~/^([^\.]+)\.(.+)$/;
	my ($name,$alias)=($1,$2);
	return "<b>[locale $fullalias]</b>" if $self->{locales_loaded}->{$locale}->{$name};
	$self->{locales_loaded}->{$locale}->{$name}=1;
	if (open IN,"templates/localization/$locale/$name.cfg") {
		while (<IN>) {
			chomp;
			Encode::_utf8_on($_);
			if (/^\s*#/) {
			} elsif (/^\s*$/) {
			} elsif (/^\s*([\w\.]+)\s*=\s*(.*?)\s*$/) {
#				warn "$1 => $2\n";
				$self->{locales}->{$locale}->{$name.".".$1}=$2;
			} else {
				warn "Unable to parse '$_'\n";
			}
		}
		close IN;
	} else {
		return "<b>Could not open templates/localization/$locale/$name.cfg</b>";
	}
	return $self->{locales}->{$locale}->{$fullalias} if $self->{locales}->{$locale}->{$fullalias};
	return "<b>[locale $fullalias]</b>";
}

###############################################################################
sub dumper {
	my ($self,$var,$level,$path)=@_;
	$level||=0;
	$path||="";
	my $path1=$path?$path."/":"";
	my $r="";
	if (ref $var eq "ARRAY") {
		$r.="ARRAY [<br>";
		my $i=0;
		my $s=@$var;
		foreach my $v (@$var) {
			$i++;
			$r.=("&nbsp;"x$level).$self->dumper($v,$level+4,"");
			$r.="," if $i<$s;
			$r.="<br>";
		}
		$r.=("&nbsp;"x($level?$level-4:0))."]";
	} elsif (ref $var eq "HASH") {
		$r.="HASH {<br>";
		foreach my $k (sort keys %$var) {
			$r.=("&nbsp;"x$level).$path1.$k." = ".$self->dumper($var->{$k},$level+4,$path1.$k)."<br>";
		}
		$r.=("&nbsp;"x($level?$level-4:0))."}";
	} else {
		Encode::_utf8_on($var);
		$r.="<span style='background-color: #e0e0e0; color: #000000;'>".$self->text($var)."</span>\n";
	}
	return $r;
}

###############################################################################

1;

