package PgWrapper;

use strict;
use DBI;
use DBD::Pg qw(:pg_types);

use utf8;
use Encode;
use Data::Dumper;
use Carp;

###############################################################################
sub new {
	my ($class,$login,$password,$database,$driver,$host)=@_;
	$driver||="Pg";

	my $dbh;
	$host||="127.0.0.1";
	if ($driver eq "Pg") {
		$dbh=DBI->connect("dbi:Pg:dbname=$database;host=$host",$login,$password,{AutoCommit=>0});
		$dbh->{pg_enable_utf8}=1;
	} elsif ($driver eq "Oracle") {
		$dbh=DBI->connect("dbi:Oracle:$database",$login,$password,{AutoCommit=>0,ora_charset => 'AL32UTF8'});
	} elsif ($driver eq "mysql") {
		$dbh=DBI->connect("dbi:mysql:database=$database;host=$host",$login,$password);
		$dbh->do("set character set utf8");
		$dbh->do("set names utf8;");
	}
	return undef unless $dbh;
	$dbh->{FetchHashKeyName} = 'NAME_lc';
	my $self={
		dbh=>$dbh,
		driver=>$driver,
		sths=>{},
		prefix=>"",
		paths=>["custom/",""]
	};
	$dbh->{LongReadLen} = 10*1024*1024;
#	$self->{dbh}->do("set character set utf8");
#	$self->{dbh}->do("set names utf8");
	$self->{dbh}->{RaiseError}=0;
	bless($self,$class);
	return $self;
}

###############################################################################
sub DESTROY {
	my ($self)=@_;
	if ($self->{dbh}) {
		$self->{dbh}->commit();
		$self->{dbh}=undef;
	}
}

###############################################################################
sub commit {
	my ($self)=@_;
	$self->{dbh}->commit();
}

###############################################################################
sub set_paths {
	my ($self,@paths)=@_;
	$self->{paths}=[@paths];
}

###############################################################################
sub last_insert_id {
	my ($self,$tablename)=@_;
	return $self->{dbh}->last_insert_id(undef,undef,$tablename,undef);
}

###############################################################################
sub read_file {
	my ($self,$file)=@_;
	my $r;
	return $self->{textfile}->{$file} if $self->{textfile}->{$file};

	foreach my $path (@{$self->{paths}}) {
		foreach my $suffix ("-".$self->{driver},"") {
			next if defined $r;
			if (open IN,$self->{prefix}.$path.$file.$suffix.".sql") {
				{local $/;$r=<IN>;}
				close IN;
			}
		}
	}
	die "PgWrapper::read_file() - Can not open $file\nSearched in (".join(",",@{$self->{paths}}).") paths\n".Carp::shortmess() unless defined $r;
	Encode::_utf8_on($r);
	my $rargs=[];
#	while ($r=~s/:([a-z][a-z0-9_]+)/\?/) {
#		push(@$rargs,$1);
#	}
#	$self->{rargs}->{$file}=$rargs;
	$r=$self->rebuild($r);
	$self->{textfile}->{$file}=$r;
	return $r;
}

###############################################################################
sub rebuild {
	my ($self,$text)=@_;
	return $text if $self->{driver} eq "Pg";
	$text=~s/coalesce\(/nvl\(/g;
	$text=~s/now\(\)\+interval '(\d+) hours'/sysdate+$1\/24/g;
	$text=~s/now\(\)/sysdate/g;
	return $text;
}

###############################################################################
sub execute {
	my ($self,$name,@args)=@_;
	unless ($self->{sths}->{$name}) {
		$self->{sths}->{$name}=$self->{dbh}->prepare($self->read_file($name));
		unless ($self->{sths}->{$name}) {
			my $err;
			$err.="Cannot prepare query $name (".$self->{dbh}->errstr().")\n";
			die $err;
		}
	}
	$self->{sths}->{$name}->execute(@args) or die "Could not execute $name (".join(",",@args).") - ".($self->{dbh}->errstr())."\n";;
	return $self->{sths}->{$name};
}

###############################################################################
sub rebuild_hash {
	my ($hash)=@_;
	foreach my $v (values %$hash) {
		Encode::_utf8_on($v);
	}
}

###############################################################################
sub execute_and_fetch {
	my ($self,$name,@args)=@_;
	my $sth=$self->execute($name,@args);
	return $self->fetch($sth);
}

###############################################################################
sub fetch {
	my ($self,$sth)=@_;
	my @arr;
	while ($_=$sth->fetchrow_hashref()) {
#		rebuild_hash($_);
		push(@arr,{(%$_)});
	}
	return @arr;
}

###############################################################################
sub execute_and_fetch_one {
	my ($self,$name,@args)=@_;
	my $sth=$self->execute($name,@args);
	$a=$sth->fetchrow_hashref();
#	rebuild_hash($a);
	return $a;
}

###############################################################################
sub execute_and_fetch_h {
	my ($self,$name,@args)=@_;
	return $self->execute_and_fetch_h_real($name,1,0,@args);
}

###############################################################################
sub execute_and_fetch_h_real {
	my ($self,$name,$level,$pid,@args)=@_;
	my @ret;
	my @tmp;
	@tmp=$self->execute_and_fetch($name,$pid,@args);
	foreach my $i (@tmp) {
		push(@ret,$i);
		$i->{level}=$level;
		push(@ret,$self->execute_and_fetch_h_real($name,$level+1,$i->{id},@args));
	}
	return @ret;
}

###############################################################################
sub execute_and_fetch_rev_h {
	my ($self,$name,$args1,$args2,$level,$id)=@_;
	$level||=1;
	my @ret;
	my @tmp;
	if ($id) {
#		warn "$name-h1 ".join(" ",$id,@$args2)."\n";
		@tmp=$self->execute_and_fetch($name."-h1",$id,@$args2);
	} else {
#		warn "$name-h0 ".join(" ",@$args1)."\n";
		@tmp=$self->execute_and_fetch($name."-h0",@$args1);
	}
	foreach my $i (@tmp) {
		push(@ret,$i);
		$i->{level}=$level;
		push(@ret,$self->execute_and_fetch_rev_h($name,$args1,$args2,$level+1,$i->{parent_id})) if $i->{parent_id};
	}
	return @ret;
}

###############################################################################
sub bind {
	my ($self,$sth,$args,$query,$name)=@_;
	my $blobs={};
	my $clobs={};
	foreach (keys %{$sth->{ParamValues}}) {
		s/^://;
		my $v;
		if ($_ eq "out_newid") {
			my $l=length($args->{$_});
			$l=20 if $l<20;
			$sth->bind_param_inout(":".$_,\ $args->{$_},$l);
			next;
		}
		if (exists $args->{$_}) {
			$v=$args->{$_};
			die "Should not bind a reference, $name - $_\n" if ref $v;
		} elsif (!$query) {
			die "bind - no \$query supplied, param $_\n\n$name\n";
		} else {
			$v=$query->param($_);
		}
		$v=undef if defined $v && $v eq "";
		if (defined $v) {
			Encode::_utf8_on($v);
		}
		if ($args->{$_."_is_blob"}) {
			$blobs->{$_}=$v;
		} elsif ($args->{$_."_is_clob"}) {
			$clobs->{$_}=$v;
		} else {
			$sth->bind_param(":".$_,$v,$args->{$_."_type"}||undef);
		}
	}
	while (my ($k,$v)=each %$blobs) {
#		warn "Blob $k\n";
		if ($self->{driver} eq "Pg") {
			$sth->bind_param(":".$k,$v,{ pg_type => DBD::Pg::PG_BYTEA });
		} elsif ($self->{driver} eq "Oracle") {
			$sth->bind_param(":".$k,$v,{ora_type=>DBD::Oracle::SQLT_BIN()});
		}
	}
	while (my ($k,$v)=each %$clobs) {
#		warn "Clob $k\n";
		if ($self->{driver} eq "Pg") {
			$sth->bind_param(":".$k,$v);
		} elsif ($self->{driver} eq "Oracle") {
			$sth->bind_param(":".$k,$v,{ora_type=>DBD::Oracle::SQLT_CHR()});
		}
	}

}

###############################################################################
sub bind_error {
	my ($self,$sth,$args,$query)=@_;
	my $err="";
	foreach (keys %{$sth->{ParamValues}}) {
		$err.="\n$_=";
		s/^://;
		my $v;
		if (exists $args->{$_}) {
			$v=$args->{$_};
		} else {
			$v=$query->param($_);
		}
		if ($args->{$_."_is_blob"}) {
			$v="[blob]";
		}
		if (defined $v) {
			Encode::_utf8_on($v);
		}
		$err.=defined $v?$v:"[undef]";
	}
	return $err;
}

###############################################################################
sub execute_bind {
	my ($self,$name,$args,$query)=@_;
	my @args;
#	die "TODO\n";
#	warn Dumper($name,$args);
	unless ($self->{sths}->{$name}) {
		$self->{sths}->{$name}=$self->{dbh}->prepare($self->read_file($name));
		unless ($self->{sths}->{$name}) {
			my $err;
			$err.="Cannot prepare query $name (".$self->{dbh}->errstr().")\n";
			die $err;
		}
	}
	my $sth=$self->{sths}->{$name};
	$self->bind($sth,$args,$query,$name);
	
	unless ($sth->execute()) {
		my $err;
		$err.="Cannot execute query $name (".$self->{dbh}->errstr().")\n";
		$err.=$self->bind_error($sth,$args,$query);
		die $err;
	}
	return $self->{sths}->{$name};
}

###############################################################################
sub execute_bind_and_fetch {
	my ($self,$name,$args,$query)=@_;
	my $sth=$self->execute_bind($name,$args,$query);
	return $self->fetch($sth);
}

###############################################################################
sub execute_bind_and_fetch_one {
	my ($self,$name,$args,$query)=@_;
	my $sth=$self->execute_bind($name,$args,$query);
	$a=$sth->fetchrow_hashref();
#	rebuild_hash($a);
	return $a;
}

###############################################################################
sub execute_bind_and_fetch_single {
	my ($self,$text,$args,$query,$hierflag)=@_;
	my $sth=$self->{dbh}->prepare($text);
	unless ($hierflag) {
		my $sth=$self->execute_bind_single($text,$args,$query);
		return $self->fetch($sth);
	}
	return $self->execute_bind_and_fetch_single_recurse($text,$args,$query,1,0);
}

###############################################################################
sub execute_bind_and_fetch_h {
	my ($self,$text,$args,$query,$level,$pid)=@_;
	$level||=1;
	my @ret;
	my @tmp;
	$args->{"parent_id"}=$pid;
	@tmp=$self->execute_bind_and_fetch($text,$args,$query);
	foreach my $i (@tmp) {
		push(@ret,$i);
		$i->{level}=$level;
		push(@ret,$self->execute_bind_and_fetch_h($text,$args,$query,$level+1,$i->{id}));
	}
	return @ret;
}

###############################################################################
sub execute_bind_and_fetch_tree {
	my ($self,$text,$args,$query,$field,$level,$pid)=@_;
	$level||=1;
	my @ret;
	$args->{"parent_id"}=$pid;
	@ret=$self->execute_bind_and_fetch($text,$args,$query);
	foreach my $i (@ret) {
		$i->{level}=$level;
		$i->{$field}=[$self->execute_bind_and_fetch_tree($text,$args,$query,$field,$level+1,$i->{id})];
	}
	return @ret;
}


###############################################################################
sub execute_bind_and_fetch_single_recurse {
	my ($self,$text,$args,$query,$level,$pid)=@_;
	my @ret;
	my @tmp;
	$args->{"parent_id"}=$pid;
	@tmp=$self->fetch($self->execute_bind_single($text,$args,$query));
	foreach my $i (@tmp) {
		push(@ret,$i);
		$i->{level}=$level;
		push(@ret,$self->execute_bind_and_fetch_single_recurse($text,$args,$query,$level+1,$i->{id}));
	}
	return @ret;
}

###############################################################################
sub execute_bind_single {
	my ($self,$text,$args,$query)=@_;
	my $sth=$self->{dbh}->prepare($text);
	unless ($sth) {
		my $err="Cannot prepare query\n";
		$err.="(".$self->{dbh}->errstr().")\n\n$text\n\n";
		die $err;
	}
	$self->bind($sth,$args,$query,$text);
	unless ($sth->execute()) {
		my $err;
		$err="Cannot execute query\n";
		$err.="(".$self->{dbh}->errstr().")\n\n$text\n";
		$err.=$self->bind_error($sth,$args,$query);
		die $err;
	}
	return $sth;
}

###############################################################################
1;
